/*
 * Decompiled with CFR 0.152.
 */
package com.boxfuse.client.core.internal.platform.local.virtualbox;

import com.boxfuse.base.coordinates.AppCoordinates;
import com.boxfuse.base.coordinates.ImageCoordinates;
import com.boxfuse.base.enums.DbType;
import com.boxfuse.base.enums.LogsType;
import com.boxfuse.base.enums.PlatformType;
import com.boxfuse.base.env.EnvVarName;
import com.boxfuse.base.env.EnvVarValue;
import com.boxfuse.base.env.EnvironmentScript;
import com.boxfuse.base.exception.BoxfuseBugException;
import com.boxfuse.base.exception.BoxfuseException;
import com.boxfuse.base.logs.Logs;
import com.boxfuse.base.logs.cloudwatchlogs.LogsLayout;
import com.boxfuse.base.port.Port;
import com.boxfuse.base.port.PortName;
import com.boxfuse.base.types.ComponentId;
import com.boxfuse.base.types.EnvName;
import com.boxfuse.base.types.IpAddress;
import com.boxfuse.base.types.TmpGB;
import com.boxfuse.base.types.Version;
import com.boxfuse.base.util.DateUtils;
import com.boxfuse.base.util.IOUtils;
import com.boxfuse.base.util.IdUtils;
import com.boxfuse.base.util.OSUtils;
import com.boxfuse.base.util.ServerUtils;
import com.boxfuse.base.util.ShellUtils;
import com.boxfuse.base.util.StringUtils;
import com.boxfuse.base.util.ThreadUtils;
import com.boxfuse.client.core.internal.cloudwatchlogs.CloudwatchLogsMockLogs;
import com.boxfuse.client.core.internal.dev.DevAgent;
import com.boxfuse.client.core.internal.diskimage.DiskImage;
import com.boxfuse.client.core.internal.diskimage.Vmdk;
import com.boxfuse.client.core.internal.platform.DevVM;
import com.boxfuse.client.core.internal.platform.Instance;
import com.boxfuse.client.core.internal.platform.InstanceMetadata;
import com.boxfuse.client.core.internal.platform.LaunchSettings;
import com.boxfuse.client.core.internal.platform.local.LocalPlatform;
import com.boxfuse.client.core.internal.platform.local.LocalPlatformHealth;
import com.boxfuse.client.core.internal.platform.local.NoOpLogs;
import com.boxfuse.client.core.internal.platform.local.SerialPortFileLogs;
import com.boxfuse.client.core.internal.platform.local.virtualbox.VirtualBoxDevVM;
import com.boxfuse.client.core.internal.platform.local.virtualbox.VirtualBoxInstallation;
import com.boxfuse.client.core.internal.platform.local.virtualbox.VirtualBoxInstance;
import com.boxfuse.client.core.internal.platform.local.virtualbox.VirtualBoxVmInfo;
import com.boxfuse.generator.config.CpuCount;
import com.boxfuse.generator.config.InstanceId;
import com.boxfuse.generator.config.LogsFilters;
import com.boxfuse.generator.config.RamMB;
import com.boxfuse.generator.image.Image;
import com.boxfuse.generator.image.MetadataImage;
import com.boxfuse.generator.inventory.Component;
import com.boxfuse.generator.repository.Repository;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VirtualBoxPlatform
extends LocalPlatform<VirtualBoxInstance> {
    private static final Logger LOGGER = LoggerFactory.getLogger(VirtualBoxPlatform.class);
    private static final List<String> RETRY_OUTPUTS = Arrays.asList("CO_E_SERVER_EXEC_FAILURE (0x80080005)", "E_ACCESSDENIED (0x80070005)", "E_FAIL (0x80004005)", "E_INVALIDARG (0x80070057)", "NS_ERROR_ABORT (0x80004004)", "NS_ERROR_FAILURE (0x80004005)", "NS_ERROR_FACTORY_NOT_REGISTERED (0x80040154)", "NS_BASE_STREAM_WOULD_BLOCK (0x80470007)", "REGDB_E_CLASSNOTREG (0x80040154)", "RPC_E_DISCONNECTED (0x80010108)", "RPC_S_CALL_CANCELLED 0x8007071A", "RPC_S_CALL_FAILED 0x800706BE", "RPC_S_SERVER_UNAVAILABLE 0x800706BA", "RPC_S_UNKNOWN_IF 0x800706B5", "VBOX_E_OBJECT_NOT_FOUND (0x80bb0001)", "VBOX_E_VM_ERROR (0x80bb0003)", "VBOX_E_FILE_ERROR (0x80bb0004)", "VBOX_E_INVALID_OBJECT_STATE (0x80bb0007)", "VERR_VD_IMAGE_READ_ONLY");
    private static final Map<String, Integer> NOT_FAILED_OUTPUTS = new HashMap<String, Integer>();
    private static final String NATNETWORK = "BoxfuseNetwork";
    private final String vboxManagePath = VirtualBoxInstallation.getVboxManage();
    private final File ipFolder;
    private final boolean natNetwork;

    public VirtualBoxPlatform(File workDir, Repository repository, DevAgent devAgent, LogsLayout logsLayout, boolean virtualBoxOK, boolean useNatNetwork) {
        super(PlatformType.VIRTUALBOX, workDir, repository, devAgent, logsLayout, virtualBoxOK);
        VirtualBoxInstallation virtualBoxInstallation = VirtualBoxInstallation.get();
        if (Boolean.TRUE.equals(useNatNetwork) && virtualBoxInstallation.getVersion().compareTo(new Version("5.1")) < 0) {
            LOGGER.warn("boxfuse.natnetwork requires VirtualBox 5.1 or newer. Reverted to NAT networking.");
            this.natNetwork = false;
        } else {
            this.natNetwork = useNatNetwork;
        }
        this.ipFolder = new File(this.workDir, "IPs");
        if (virtualBoxInstallation.getHealth() == LocalPlatformHealth.OK) {
            ShellUtils.mkdir(this.ipFolder);
        }
    }

    @Override
    public void ensureHealthy() {
        VirtualBoxInstallation installationDetails = VirtualBoxInstallation.get();
        switch (installationDetails.getHealth()) {
            case NOT_INSTALLED: {
                throw new BoxfuseException("VirtualBox is not installed (Unable to find " + VirtualBoxInstallation.EXECUTABLE + ") => download and install the latest version from https://www.virtualbox.org/");
            }
            case BROKEN: {
                throw new BoxfuseException("VirtualBox is installed but seems to have a problem" + (installationDetails.getRawVersion() == null ? ". " : ":\n" + installationDetails.getRawVersion() + "\n") + "Download and reinstall the latest version from https://www.virtualbox.org/");
            }
            case OUTDATED: {
                throw new BoxfuseException("Your VirtualBox installation is outdated (" + installationDetails.getRawVersion() + "). Boxfuse only supports VirtualBox " + VirtualBoxInstallation.MINIMUM_SUPPORTED_VERSION + " or newer. Download and install the latest version from https://www.virtualbox.org/");
            }
        }
    }

    @Override
    public boolean needsDevAgent() {
        return false;
    }

    boolean isNatNetwork() {
        return this.natNetwork;
    }

    @Override
    protected DiskImage createOrGetDiskImage(Image image, File instanceDir) {
        return new Vmdk(image, instanceDir);
    }

    @Override
    protected InstanceId generateInstanceId() {
        return new InstanceId(IdUtils.generateLocallyUniqueId("vb"));
    }

    @Override
    public boolean isHealthy() {
        return VirtualBoxInstallation.get().getHealth().equals((Object)LocalPlatformHealth.OK);
    }

    @Override
    protected VirtualBoxInstance startAndTagInstance(InstanceId instanceId, File instanceDir, Image image, DiskImage diskImage, LaunchSettings launchSettings, Map<PortName, Port> ports, Map<EnvVarName, EnvVarValue> envVars, Logs appLogs, Date launchTime) {
        TmpGB effectiveTmp;
        Logs bootLogs;
        if (this.natNetwork && image.getBoxfuseVersion().compareTo(new Version("1")) > 0 && image.getBoxfuseVersion().compareTo(new Version("1.21")) < 0) {
            throw new BoxfuseException("Unable to run " + image.getCoordinates() + " as it was generated with an older version of Boxfuse that did not support VirtualBox NAT Networks\n=> regenerate the image with this version of Boxfuse\n=> or set virtualbox.natnetwork to false");
        }
        String vmDescription = image.getCoordinates().toString();
        boolean legacyNic = image.getComponents().get((Object)ComponentId.LINUX).compareTo(new Version("4.0")) < 0;
        this.createVM(instanceId.getId(), vmDescription, launchSettings.getCpus(), launchSettings.getRamInMB(), legacyNic);
        if (image.isLive()) {
            this.createSharedFolder(instanceId.getId(), "boxfuse", image.getPayload().getFile(), true);
        }
        if (launchSettings.isLogsBoot()) {
            File logDir = launchSettings.getLogsDir() == null ? instanceDir : new File(launchSettings.getLogsDir());
            SerialPortFileLogs serialPortFileLogs = SerialPortFileLogs.createInDir(logDir, instanceId);
            this.executeVBoxManageCommand("modifyvm", instanceId.getId(), "--uart1", "0x3F8", "4", "--uartmode1", "file", serialPortFileLogs.getLogsFilePath());
            bootLogs = serialPortFileLogs;
        } else {
            bootLogs = NoOpLogs.INSTANCE;
        }
        Map<PortName, Port> desiredLocalPorts = launchSettings.getPortsMap();
        TreeMap<PortName, Port> localPorts = new TreeMap<PortName, Port>();
        TreeMap<String, Integer> portsMapMetaData = new TreeMap<String, Integer>();
        HashSet<Port> unusablePorts = new HashSet();
        if (this.natNetwork) {
            unusablePorts = this.getUnusableLocalPorts();
        }
        for (PortName portName : ports.keySet()) {
            Port port = ports.get(portName);
            Port desiredLocalPort = desiredLocalPorts.keySet().contains(portName) ? desiredLocalPorts.get(portName) : (!OSUtils.runningOnWindows() && !ShellUtils.isRoot() && port.getNumber() < 1024 ? Port.of(10000 + port.getNumber() + "/" + (Object)((Object)port.getProtocol())) : port);
            Port localPort = Port.findFreePort(desiredLocalPort.getNumber(), port.getProtocol(), unusablePorts);
            localPorts.put(portName, localPort);
            unusablePorts.add(localPort);
            portsMapMetaData.put(portName.getName(), localPort.getNumber());
        }
        if (localPorts.containsKey(PortName.HTTP) && localPorts.containsKey(PortName.HTTPS)) {
            envVars.put(EnvVarName.BOXFUSE_PORTS_HTTPS__REDIRECT, EnvVarValue.asIs("" + ((Port)localPorts.get(PortName.HTTPS)).getNumber()));
        }
        String storageCtlName = this.computeStorageCtlName(instanceId.getId());
        this.createStorageController(instanceId.getId(), 3);
        this.attachHdd(instanceId.getId(), storageCtlName, diskImage.getMainFile(), 0);
        TmpGB tmpGB = effectiveTmp = launchSettings.getTmp() == null ? image.getTmp() : launchSettings.getTmp();
        if (effectiveTmp.getValue() > 0) {
            File tmpDiskFile = this.getTmpDiskFile(instanceDir);
            int tmpInMB = effectiveTmp.getValue() * 1024;
            this.createHdd(tmpDiskFile, tmpInMB);
            this.attachHdd(instanceId.getId(), storageCtlName, tmpDiskFile, 1);
        }
        File environmentIso = this.createEnvironmentIso(instanceDir, new EnvironmentScript(envVars));
        this.executeVBoxManageCommand("storageattach", instanceId.getId(), "--storagectl", storageCtlName, "--type", "dvddrive", "--medium", environmentIso.getAbsolutePath(), "--port", "2", "--device", "0");
        InstanceMetadata metadata = new InstanceMetadata();
        metadata.image = image.getMetadata();
        metadata.cpus = launchSettings.getCpus();
        metadata.ram = launchSettings.getRamInMB();
        metadata.logs = launchSettings.isLogsBoot();
        metadata.launchTime = DateUtils.toUtcDateTimeString(launchTime);
        metadata.portsMap = portsMapMetaData;
        this.executeVBoxManageCommand("setextradata", instanceId.getId(), "boxfuse", VirtualBoxPlatform.encodeMetadata(metadata));
        IpAddress ip = this.startVM(instanceId.getId(), true);
        if (this.natNetwork) {
            this.discardObsoleteNatNetworkPortForwardingRules();
        }
        for (PortName portName : ports.keySet()) {
            Port port = ports.get(portName);
            Port localPort = (Port)localPorts.get(portName);
            LOGGER.info("Forwarding " + portName + " port localhost:" + localPort + " -> " + instanceId + ":" + port.getNumber());
            if (this.natNetwork) {
                this.forwardNatNetworkPort(instanceId.getId(), portName.getName(), ip, port, localPort);
                continue;
            }
            this.forwardNatPort(instanceId.getId(), portName.getName(), port, localPort);
        }
        return new VirtualBoxInstance(this, !this.natNetwork, image, diskImage, instanceId, launchSettings.getCpus(), launchSettings.getRamInMB(), launchTime, bootLogs, appLogs, instanceDir, localPorts, metadata.image.payload.path);
    }

    Set<Port> getUnusableLocalPorts() {
        return VirtualBoxPlatform.extractUnusablePorts(this.listNatNetworks());
    }

    void forwardNatNetworkPort(String vmName, String portName, IpAddress ip, Port port, Port localPort) {
        this.executeVBoxManageCommand("natnetwork", "modify", "--netname", NATNETWORK, "--port-forward-4", VirtualBoxPlatform.createVirtualBoxForwardingRule(vmName, portName, ip, port, localPort));
    }

    static String createVirtualBoxForwardingRule(String vmName, String portName, IpAddress ip, Port port, Port localPort) {
        return vmName + "_" + portName + ":" + (port.getProtocol().isTcp() ? "tcp" : "udp") + ":[]:" + localPort.getNumber() + ":[" + ip + "]:" + port.getNumber();
    }

    @Override
    protected File getTmpDiskFile(File instanceDir) {
        return new File(instanceDir.getAbsolutePath() + "/tmp.disk");
    }

    private void createHdd(File hddFile, int sizeInMB) {
        this.executeVBoxManageCommand("createhd", "--filename", hddFile.getAbsolutePath(), "--size", "" + sizeInMB);
    }

    void discardObsoleteNatNetworkPortForwardingRules() {
        List<String> runningVMs = this.getRunningVMs();
        List<String> rules = VirtualBoxPlatform.extractNatNetworkPortForwardingRuleNames(this.listNatNetworks());
        for (String rule : rules) {
            if (!rule.contains("_") || runningVMs.contains(rule.substring(0, rule.indexOf("_")))) continue;
            LOGGER.debug("Discarding obsolete port forwarding rule " + rule + " ...");
            this.executeVBoxManageCommand("natnetwork", "modify", "--netname", NATNETWORK, "--port-forward-4", "delete", rule);
        }
    }

    private String listNatNetworks() {
        return this.executeVBoxManageCommand("list", "natnets");
    }

    private String listDhcpServers() {
        return this.executeVBoxManageCommand("list", "dhcpservers");
    }

    static List<String> extractNatNetworkPortForwardingRuleNames(String listNatnetsOutput) {
        ArrayList<String> result = new ArrayList<String>();
        for (String line : StringUtils.tokenizeToStringArray(listNatnetsOutput, "\n")) {
            if (!line.matches(".*:[tcudp]{3}:.*:.*:.*:.*")) continue;
            result.add(line.substring(0, line.indexOf(":")));
        }
        return result;
    }

    static Set<Port> extractUnusablePorts(String listNatnetsOutput) {
        HashSet<Port> result = new HashSet<Port>();
        for (String line : StringUtils.tokenizeToStringArray(listNatnetsOutput, "\n")) {
            if (!line.matches(".*:[tcudp]{3}:.*:.*:.*:.*")) continue;
            String[] parts = StringUtils.tokenizeToStringArray(line, ":");
            result.add(Port.of(parts[3] + "/" + parts[1]));
        }
        return result;
    }

    static Port extractForwardedPort(String listNatnetsOutput, IpAddress ip, Port port) {
        for (String line : StringUtils.tokenizeToStringArray(listNatnetsOutput, "\n")) {
            if (!line.matches(".*:" + Pattern.quote(port.getProtocol().toString()) + ":.*:.*:" + Pattern.quote("[" + ip + "]") + ":" + Pattern.quote("" + port.getNumber()))) continue;
            String[] parts = StringUtils.tokenizeToStringArray(line, ":");
            return Port.of(parts[3] + "/" + (port.getProtocol().isTcp() ? "tcp" : "udp"));
        }
        return null;
    }

    Port getNatNetworkForwardedPort(IpAddress ip, Port port) {
        return VirtualBoxPlatform.extractForwardedPort(this.listNatNetworks(), ip, port);
    }

    private void createSharedFolder(String vmName, String sharedFolderName, File hostPath, boolean readonly) {
        ArrayList<String> params = new ArrayList<String>(Arrays.asList("sharedfolder", "add", vmName, "--name", sharedFolderName, "--hostpath", hostPath.getAbsolutePath()));
        if (readonly) {
            params.add("--readonly");
        }
        this.executeVBoxManageCommand(params);
    }

    void attachHdd(String instanceId, String storageCtlName, File hddFile, int sataPortNum) {
        this.executeVBoxManageCommand("storageattach", instanceId, "--storagectl", storageCtlName, "--type", "hdd", "--medium", hddFile.getAbsolutePath(), "--port", "" + sataPortNum, "--device", "0");
    }

    private void detachHdd(String instanceId, String storageCtlName, int sataPortNum) {
        this.executeVBoxManageCommand("storageattach", instanceId, "--storagectl", storageCtlName, "--type", "hdd", "--medium", "none", "--port", "" + sataPortNum, "--device", "0");
    }

    private void detachDvd(String instanceId, String storageCtlName, int sataPortNum) {
        this.executeVBoxManageCommand("storageattach", instanceId, "--storagectl", storageCtlName, "--type", "dvddrive", "--medium", "emptydrive", "--port", "" + sataPortNum, "--device", "0");
    }

    void clearNatForwardedPort(String vmName, String portName) {
        this.executeVBoxManageCommand("modifyvm", vmName, "--natpf1", "delete", portName);
    }

    void createVM(String vmName, String vmDescription, CpuCount cpus, RamMB ramInMB, boolean legacyNic) {
        this.checkEnvironment();
        if (this.natNetwork) {
            if (this.listNatNetworks().contains(NATNETWORK)) {
                this.executeVBoxManageCommand("natnetwork", "modify", "--netname", NATNETWORK, "--enable", "--dhcp", "on", "--ipv6", "off");
            } else {
                this.executeVBoxManageCommand("natnetwork", "add", "--netname", NATNETWORK, "--network", "192.168.15.0/24", "--enable", "--dhcp", "on", "--ipv6", "off");
            }
            if (this.listDhcpServers().contains(NATNETWORK)) {
                this.executeVBoxManageCommand("dhcpserver", "modify", "--netname", NATNETWORK, "--enable", "--ip", "192.168.15.3", "-netmask", "255.255.255.0", "--lowerip", "192.168.15.4", "--upperip", "192.168.15.254");
            } else {
                this.executeVBoxManageCommand("dhcpserver", "add", "--netname", NATNETWORK, "--enable", "--ip", "192.168.15.3", "-netmask", "255.255.255.0", "--lowerip", "192.168.15.4", "--upperip", "192.168.15.254");
            }
            this.executeVBoxManageCommand("natnetwork", "start", "--netname", NATNETWORK);
        }
        this.executeVBoxManageCommand("createvm", "--name", vmName, "--register", "--basefolder", this.workDir.getAbsolutePath());
        ArrayList<String> modifyVmParams = new ArrayList<String>();
        modifyVmParams.addAll(Arrays.asList("modifyvm", vmName));
        modifyVmParams.addAll(this.getModifyVmDescriptionParams(vmDescription));
        modifyVmParams.addAll(Arrays.asList("--ostype", "Linux26_64", "--cpus", "" + cpus, "--memory", "" + ramInMB, "--vram", "20", "--rtcuseutc", "on", "--acpi", "on", "--ioapic", "on", "--audio", "none", "--vrde", "off", "--nic1", !this.natNetwork ? "nat" : "natnetwork", "--nictype1", legacyNic ? "virtio" : "82545EM", "--cableconnected1", "on", "--boot1", "disk", "--boot2", "none", "--boot3", "none", "--boot4", "none", "--hwvirtex", "on", "--nestedpaging", "on", "--vtxvpid", "on", "--largepages", "on", "--bioslogodisplaytime", "1"));
        if (this.natNetwork) {
            modifyVmParams.addAll(Arrays.asList("--nat-network1", NATNETWORK));
        }
        this.executeVBoxManageCommand(modifyVmParams);
        File ipFile = this.getIpFile(vmName);
        if (ipFile.exists()) {
            ShellUtils.deleteFile(ipFile);
        }
        this.createSharedFolder(vmName, "boxfuse-IPs", this.ipFolder, false);
    }

    void updateHddUuid(File hdd) {
        if (!hdd.setWritable(true)) {
            LOGGER.warn("Unable to make Dev VM HDD writable");
        }
        this.executeVBoxManageCommand("internalcommands", "sethduuid", hdd.getAbsolutePath());
    }

    private File getIpFile(String vmName) {
        return new File(this.ipFolder, vmName + ".ip");
    }

    IpAddress startVM(String vmName, boolean waitForIp) {
        try {
            this.executeVBoxManageCommand("startvm", vmName, "--type", "headless");
        }
        catch (BoxfuseException e) {
            if (!this.isPlatformOK()) {
                String message = "Unable to start VirtualBox instance " + vmName;
                message = message + "\n=> ensure VirtualBox is working correctly\n=> ensure hardware virtualization (VT-x or AMD-V) is enabled on your machine and check by running \"" + this.vboxManagePath + " list hostinfo\"";
                if (OSUtils.runningOnWindows()) {
                    message = message + "\n=> ensure Hyper-V is disabled";
                }
                LOGGER.debug(e.getMessage());
                throw new BoxfuseException(message);
            }
            throw e;
        }
        if (this.natNetwork && waitForIp) {
            File ipFile = this.getIpFile(vmName);
            String ip = IOUtils.copyToString(ipFile);
            while (!(ipFile.isFile() && ipFile.canRead() && StringUtils.hasLength(ip))) {
                ThreadUtils.sleep(500L);
                ip = IOUtils.copyToString(ipFile);
            }
            return IpAddress.of(ip);
        }
        return IpAddress.of("10.0.2.15");
    }

    private void checkEnvironment() {
        if ("10.0.2.15".equals(ServerUtils.getHostIpAddress().toString())) {
            throw new BoxfuseException("Unable to run VirtualBox inside a VirtualBox VM due to VirtualBox limitations (https://www.virtualbox.org/ticket/4032) => run Boxfuse on your host OS instead.");
        }
    }

    @Override
    public List<Instance> runningInstances() {
        ArrayList<Instance> instances = new ArrayList<Instance>();
        List<String> vmNames = this.getRunningVMs();
        for (String vmName : vmNames) {
            Instance instance;
            if (!InstanceId.isValid(vmName) || (instance = this.doGetInstance(new InstanceId(vmName))) == null) continue;
            instances.add(instance);
        }
        Collections.sort(instances);
        return instances;
    }

    @Override
    public List<Instance> getInstances(Collection<InstanceId> instanceIds) {
        ArrayList<Instance> result = new ArrayList<Instance>();
        List<String> runningInstanceIds = this.getRunningVMs();
        for (InstanceId instanceId : instanceIds) {
            if (!runningInstanceIds.contains(instanceId.getId())) {
                throw new BoxfuseException("No Instance with id '" + instanceId + "' found in the dev environment => ensure the instance id and the environment are correct");
            }
            Instance instance = this.doGetInstance(instanceId);
            if (instance == null) {
                throw new BoxfuseException("Instance with id '" + instanceId + "' is not a boxfuse Instance!");
            }
            result.add(instance);
        }
        return result;
    }

    private Instance doGetInstance(InstanceId instanceId) {
        Logs appLogs;
        Logs bootLogs;
        String extraData = this.executeVBoxManageCommand("getextradata", instanceId.getId(), "boxfuse");
        if (!extraData.startsWith("Value: ")) {
            return null;
        }
        String encodedTags = extraData.substring("Value: ".length());
        InstanceMetadata metadata = VirtualBoxPlatform.decodeMetadata(encodedTags);
        if (metadata == null) {
            LOGGER.warn("Unable to decode metadata for " + instanceId);
            return null;
        }
        if (metadata.image.owner == null) {
            LOGGER.warn("Invalid metadata for " + instanceId);
            LOGGER.trace("Metadata: " + encodedTags);
            return null;
        }
        Image image = this.repository.findLocally(new ImageCoordinates(metadata.image.owner, metadata.image.name, metadata.image.version));
        if (image == null) {
            image = new MetadataImage(metadata.image);
        }
        File instanceDir = this.getInstanceDir(instanceId);
        if (metadata.logs.booleanValue()) {
            File logDir = metadata.logsDir == null ? instanceDir : new File(metadata.logsDir);
            bootLogs = SerialPortFileLogs.createInDir(logDir, instanceId);
        } else {
            bootLogs = NoOpLogs.INSTANCE;
        }
        HashMap<PortName, Port> localPorts = new HashMap<PortName, Port>();
        for (String localPort : metadata.portsMap.keySet()) {
            localPorts.put(PortName.of(localPort), Port.getUnrestrictedTcpPort((Integer)metadata.portsMap.get(localPort)));
        }
        if (LogsType.CLOUDWATCH_LOGS == image.getLogsType()) {
            LogsFilters filters = new LogsFilters(this.repository.getOwner());
            filters.addAppFilter(image.getCoordinates().getAppCoordinates());
            filters.addInstanceFilter(instanceId);
            appLogs = new CloudwatchLogsMockLogs(this.repository.getOwner(), this.devAgent, EnvName.DEV, filters, this.logsLayout);
        } else {
            appLogs = NoOpLogs.INSTANCE;
        }
        Vmdk diskImage = new Vmdk(image, instanceDir);
        return new VirtualBoxInstance(this, !this.natNetwork, image, diskImage, instanceId, metadata.cpus, metadata.ram, DateUtils.toUtcDateTime(metadata.launchTime), bootLogs, appLogs, instanceDir, localPorts, metadata.image.payload.path);
    }

    static List<String> extractInstanceIds(String listCommandOutput) {
        ArrayList<String> ids = new ArrayList<String>();
        if (StringUtils.hasLength(listCommandOutput)) {
            Pattern pattern = Pattern.compile("\"(.+)\"");
            Matcher matcher = pattern.matcher(listCommandOutput);
            while (matcher.find()) {
                ids.add(matcher.group(1));
            }
        }
        return ids;
    }

    private String executeVBoxManageCommand(List<String> args2) {
        return this.executeVBoxManageCommand(args2.toArray(new String[args2.size()]));
    }

    String executeVBoxManageCommand(String ... args2) {
        return ShellUtils.execCommandWithRetries(3, RETRY_OUTPUTS, NOT_FAILED_OUTPUTS, this.vboxManagePath, args2);
    }

    private String executeVBoxManageCommand(Map<String, Integer> notFailedOUtputs, String ... args2) {
        return ShellUtils.execCommandWithRetries(3, RETRY_OUTPUTS, notFailedOUtputs, this.vboxManagePath, args2);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        VirtualBoxPlatform that = (VirtualBoxPlatform)o;
        return this.vboxManagePath.equals(that.vboxManagePath);
    }

    @Override
    public int hashCode() {
        return this.vboxManagePath.hashCode();
    }

    List<String> getRunningVMs() {
        return VirtualBoxPlatform.extractInstanceIds(this.executeVBoxManageCommand("list", "runningvms"));
    }

    VirtualBoxVmInfo getVMInfo(String vmName) {
        return new VirtualBoxVmInfo(this.executeVBoxManageCommand("showvminfo", vmName, "--machinereadable"));
    }

    File computeHddFile(String vmName) {
        File vmDir = new File(this.workDir, vmName);
        return new File(vmDir, "disk.vmdk");
    }

    String computeStorageCtlName(String vmName) {
        return vmName + "-StorageCtl";
    }

    String forwardNatPort(String vmName, String portName, Port port, Port localPort) {
        return this.executeVBoxManageCommand("controlvm", vmName, "natpf1", VirtualBoxPlatform.createNatRule(portName, port, localPort));
    }

    static String createNatRule(String portName, Port port, Port localPort) {
        return portName + "," + (port.getProtocol().isTcp() ? "tcp" : "udp") + ",," + localPort.getNumber() + ",," + port.getNumber();
    }

    void destroyVM(String vmName) {
        HashMap<String, Integer> notFailedOutputs = new HashMap<String, Integer>();
        notFailedOutputs.put("VERR_PATH_NOT_FOUND", 1);
        notFailedOutputs.put("Machine delete failed", 1);
        notFailedOutputs.put("Could not find a registered machine named", 1);
        while (this.vmExists(vmName)) {
            ShellUtils.execCommandWithRetries(3, RETRY_OUTPUTS, notFailedOutputs, this.vboxManagePath, "unregistervm", vmName, "--delete");
            ThreadUtils.sleep(1000L);
        }
        this.deleteVmDir(vmName);
    }

    @Override
    protected boolean vmExists(String vmName) {
        return VirtualBoxPlatform.extractInstanceIds(this.executeVBoxManageCommand("list", "vms")).contains(vmName);
    }

    void closeAndDeleteHdd(String vmName, String storageCtlName, File diskFile, int port) {
        try {
            if (diskFile.exists()) {
                this.detachHdd(vmName, storageCtlName, port);
                this.deleteHdd(diskFile);
            }
        }
        catch (Exception e) {
            LOGGER.warn("Unable to delete disk for " + vmName);
            LOGGER.debug("VirtualBox Error: " + e.getMessage());
        }
    }

    void closeAndDeleteDvd(String vmName, String storageCtlName, File dvdFile) {
        try {
            if (dvdFile.exists()) {
                this.detachDvd(vmName, storageCtlName, 2);
                this.deleteDvd(dvdFile);
            }
        }
        catch (Exception e) {
            LOGGER.warn("Unable to delete envvars disc for " + vmName);
            LOGGER.debug("VirtualBox Error: " + e.getMessage());
        }
    }

    private void deleteHdd(File diskFile) {
        this.executeVBoxManageCommand("closemedium", "disk", diskFile.getAbsolutePath(), "--delete");
    }

    private void deleteDvd(File dvdFile) {
        this.executeVBoxManageCommand("closemedium", "dvd", dvdFile.getAbsolutePath(), "--delete");
    }

    void killVM(String vmName) {
        HashMap<String, Integer> notFailed = new HashMap<String, Integer>(NOT_FAILED_OUTPUTS);
        notFailed.put("not currently running", 1);
        this.executeVBoxManageCommand(notFailed, "controlvm", vmName, "acpipowerbutton");
        if (this.waitForPoweroff(vmName)) {
            return;
        }
        LOGGER.warn("Timeout waiting for " + vmName + " to terminate. Current state: " + VirtualBoxPlatform.getState(this.getVmInfo(vmName)) + " => retrying with direct poweroff ...");
        this.executeVBoxManageCommand(notFailed, "controlvm", vmName, "poweroff");
        if (this.waitForPoweroff(vmName)) {
            return;
        }
        throw new BoxfuseException("Timeout waiting for " + vmName + " to power off. Current state: " + VirtualBoxPlatform.getState(this.getVmInfo(vmName)));
    }

    private boolean waitForPoweroff(String vmName) {
        long start = System.currentTimeMillis();
        while (!"poweroff".equals(VirtualBoxPlatform.getState(this.getVmInfo(vmName)))) {
            if (System.currentTimeMillis() - 15000L > start) {
                return false;
            }
            ThreadUtils.sleep(1000L);
        }
        return true;
    }

    static String getState(String vmInfo) {
        Matcher matcher = Pattern.compile(".*VMState=\"([a-z]+)\".*").matcher(vmInfo);
        if (!matcher.find()) {
            throw new BoxfuseBugException("Unable to parse VirtualBox VM info: " + vmInfo);
        }
        return matcher.group(1);
    }

    private String getVmInfo(String vmName) {
        return this.executeVBoxManageCommand("showvminfo", vmName, "--machinereadable");
    }

    @Override
    public boolean supportsDevVM() {
        return true;
    }

    @Override
    public DevVM getDevVM(Component devHddComponent, AppCoordinates appCoordinates, DbType dbType) {
        return new VirtualBoxDevVM(this, devHddComponent, appCoordinates, dbType);
    }

    IpAddress getVmIpUsingGuestProperty(String vmName) {
        String output = this.executeVBoxManageCommand("guestproperty", "get", vmName, "/VirtualBox/GuestInfo/Net/0/V4/IP");
        if (!output.startsWith("Value: ")) {
            throw new BoxfuseException("Unable to determine Dev VM IP address: " + output + "\n=> check VirtualBox and ensure the " + vmName + " VM came up correctly");
        }
        return IpAddress.of(output.substring("Value: ".length()).trim());
    }

    private List<String> getModifyVmDescriptionParams(String description) {
        return Arrays.asList("--description", description);
    }

    void createStorageController(String instanceId, int portCount) {
        String storageCtlName = this.computeStorageCtlName(instanceId);
        this.executeVBoxManageCommand("storagectl", instanceId, "--name", storageCtlName, "--add", "sata", "--controller", "IntelAHCI", "--portcount", "" + portCount, "--hostiocache", "on", "--bootable", "on");
    }

    static {
        NOT_FAILED_OUTPUTS.put("has been successfully started.", 1);
        NOT_FAILED_OUTPUTS.put("NATNetwork server already exists", 2);
        NOT_FAILED_OUTPUTS.put("RemovePortForwardRule", 2);
    }
}

