/*
 * Decompiled with CFR 0.152.
 */
package com.boxfuse.base.util;

import com.boxfuse.base.exception.BoxfuseBugException;
import com.boxfuse.base.exception.BoxfuseException;
import com.boxfuse.base.payload.PayloadEntry;
import com.boxfuse.base.util.IOUtils;
import com.boxfuse.base.util.ShellUtils;
import com.boxfuse.base.util.StringUtils;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.xz.XZCompressorInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZipUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(ZipUtils.class);

    private ZipUtils() {
    }

    public static byte[] zipDirContents(File dir) {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        ZipOutputStream zos = new ZipOutputStream(os);
        try {
            File[] files = dir.listFiles();
            if (files != null) {
                for (File file : files) {
                    ZipUtils.addRecursivelyToZipArchive(zos, file, null);
                }
            }
        }
        catch (IOException e) {
            throw new BoxfuseException("Unable to zip directory: " + dir.getAbsolutePath(), e);
        }
        finally {
            IOUtils.close(zos);
        }
        return os.toByteArray();
    }

    private static void addRecursivelyToZipArchive(ZipOutputStream zos, File fileToZip, String parentDirectoryName) throws IOException {
        if (fileToZip == null || !fileToZip.exists()) {
            return;
        }
        String zipEntryName = fileToZip.getName();
        if (parentDirectoryName != null && !parentDirectoryName.isEmpty()) {
            zipEntryName = parentDirectoryName + "/" + fileToZip.getName();
        }
        if (fileToZip.isDirectory()) {
            File[] files = fileToZip.listFiles();
            if (files != null) {
                for (File file : files) {
                    ZipUtils.addRecursivelyToZipArchive(zos, file, zipEntryName);
                }
            }
        } else {
            FileInputStream fis = new FileInputStream(fileToZip);
            zos.putNextEntry(new ZipEntry(zipEntryName));
            IOUtils.copyNoClose(fis, (OutputStream)zos);
            zos.closeEntry();
            fis.close();
        }
    }

    public static boolean isZipFile(byte[] bytes) {
        try {
            new ZipInputStream(new ByteArrayInputStream(bytes)).close();
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isZipFile(File file) {
        try (ZipFile ignored = new ZipFile(file);){
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            return false;
        }
    }

    public static Map<String, PayloadEntry> listEntries(byte[] bytes) {
        TreeMap<String, PayloadEntry> entries = new TreeMap<String, PayloadEntry>();
        ZipInputStream zipInputStream = null;
        try {
            ZipEntry zipEntry;
            zipInputStream = new ZipInputStream(new ByteArrayInputStream(bytes));
            while ((zipEntry = zipInputStream.getNextEntry()) != null) {
                if (zipEntry.isDirectory()) continue;
                entries.put(zipEntry.getName(), new PayloadEntry(zipEntry.getName(), false));
            }
        }
        catch (IOException e) {
            try {
                throw new BoxfuseException("Unable to read zip file (" + e.getMessage() + ") => ensure it isn't corrupt and try again", e);
            }
            catch (Throwable throwable) {
                IOUtils.close(zipInputStream);
                throw throwable;
            }
        }
        IOUtils.close(zipInputStream);
        return entries;
    }

    public static Map<String, PayloadEntry> listTarGzEntries(byte[] tarGz) {
        TreeMap<String, PayloadEntry> names = new TreeMap<String, PayloadEntry>();
        TarArchiveInputStream tar = null;
        try {
            TarArchiveEntry entry;
            tar = new TarArchiveInputStream(new GZIPInputStream(new ByteArrayInputStream(tarGz)));
            while ((entry = (TarArchiveEntry)tar.getNextEntry()) != null) {
                if (entry.isDirectory()) continue;
                names.put(entry.getName(), new PayloadEntry(entry.getName(), ZipUtils.isExecutable(entry.getMode())));
            }
        }
        catch (IOException e) {
            try {
                throw new BoxfuseException("Unable to read tar.gz file (" + e.getMessage() + ")", e);
            }
            catch (Throwable throwable) {
                IOUtils.close(tar);
                throw throwable;
            }
        }
        IOUtils.close(tar);
        return names;
    }

    static boolean isExecutable(int mode) {
        return (mode & 0x49) != 0;
    }

    public static boolean containsEntryMatching(byte[] bytes, String regex) {
        for (String name : ZipUtils.listEntries(bytes).keySet()) {
            if (!name.matches(regex)) continue;
            return true;
        }
        return false;
    }

    public static void close(ZipFile zipfile) {
        try {
            if (zipfile != null) {
                zipfile.close();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static byte[] unzipSingleEntryToByteArray(File sourceZipFile, String sourceEntryName) {
        byte[] byArray;
        ZipFile zipFile = null;
        InputStream in = null;
        try {
            zipFile = new ZipFile(sourceZipFile);
            ZipEntry entry = zipFile.getEntry(sourceEntryName);
            in = zipFile.getInputStream(entry);
            byArray = IOUtils.copyToByteArray(in);
        }
        catch (IOException e) {
            try {
                throw new BoxfuseException("Unable to extract " + sourceEntryName + " from " + sourceZipFile.getAbsolutePath(), e);
            }
            catch (Throwable throwable) {
                IOUtils.close(in);
                ZipUtils.close(zipFile);
                throw throwable;
            }
        }
        IOUtils.close(in);
        ZipUtils.close(zipFile);
        return byArray;
    }

    public static byte[] unzipSingleEntryToByteArray(byte[] sourceZipBytes, String sourceEntryName) {
        ZipInputStream zipStream = null;
        try {
            ZipEntry entry;
            zipStream = new ZipInputStream(new ByteArrayInputStream(sourceZipBytes));
            while ((entry = zipStream.getNextEntry()) != null) {
                if (!sourceEntryName.equals(entry.getName())) continue;
                byte[] byArray = IOUtils.copyToByteArrayNoClose(zipStream);
                return byArray;
            }
        }
        catch (IOException e) {
            throw new BoxfuseException("Unable to extract zip file entry: " + sourceEntryName, e);
        }
        finally {
            if (zipStream != null) {
                try {
                    zipStream.close();
                }
                catch (IOException iOException) {}
            }
        }
        return null;
    }

    public static String unzipSingleEntry(File sourceZipFile, String sourceEntryName) {
        String string;
        ZipFile zipFile = null;
        InputStream in = null;
        try {
            zipFile = new ZipFile(sourceZipFile);
            ZipEntry entry = zipFile.getEntry(sourceEntryName);
            if (entry == null) {
                throw new BoxfuseException("Unable to find " + sourceEntryName + " in " + sourceZipFile.getAbsolutePath());
            }
            in = zipFile.getInputStream(entry);
            string = IOUtils.copyToString(in, (int)entry.getSize());
        }
        catch (IOException e) {
            try {
                throw new BoxfuseException("Unable to extract " + sourceEntryName + " from " + sourceZipFile.getAbsolutePath(), e);
            }
            catch (Throwable throwable) {
                IOUtils.close(in);
                ZipUtils.close(zipFile);
                throw throwable;
            }
        }
        IOUtils.close(in);
        ZipUtils.close(zipFile);
        return string;
    }

    public static String unzipSingleEntry(byte[] sourceZipBytes, String sourceEntryName) {
        return StringUtils.toUtf8String(ZipUtils.unzipSingleEntryToByteArray(sourceZipBytes, sourceEntryName));
    }

    public static InputStream unzipSingleEntryToStream(InputStream sourceZipStream, final String sourceEntryName) {
        return ZipUtils.unzipSingleEntryToStream(sourceZipStream, new EntryFilter(){

            @Override
            public boolean matches(String entry) {
                return entry.equals(sourceEntryName);
            }
        });
    }

    public static InputStream unzipSingleEntryToStream(final InputStream sourceZipStream, EntryFilter entryFilter) {
        try {
            ZipEntry entry;
            ZipInputStream zipStream = new ZipInputStream(sourceZipStream){

                @Override
                public void close() throws IOException {
                    while (this.getNextEntry() != null) {
                    }
                    while (this.read() >= 0) {
                    }
                    while (sourceZipStream.read() >= 0) {
                    }
                    super.close();
                }
            };
            while ((entry = zipStream.getNextEntry()) != null) {
                if (!entryFilter.matches(entry.getName())) continue;
                return zipStream;
            }
        }
        catch (IOException e) {
            throw new BoxfuseException("Unable to stream zip file entry", e);
        }
        return null;
    }

    public static void unzip(byte[] sourceZipBytes, File targetDir) {
        try {
            ZipEntry entry;
            ZipInputStream zipStream = new ZipInputStream(new ByteArrayInputStream(sourceZipBytes));
            while ((entry = zipStream.getNextEntry()) != null) {
                File entryFile = new File(targetDir + "/" + entry.getName());
                if (entry.isDirectory()) {
                    ShellUtils.mkdir(entryFile.getPath());
                    continue;
                }
                ShellUtils.mkdir(entryFile.getParent());
                FileOutputStream out = new FileOutputStream(entryFile);
                IOUtils.copyNoClose(zipStream, (OutputStream)out);
                out.close();
                LOGGER.debug("Unzipped " + entry.getName());
                zipStream.closeEntry();
            }
            zipStream.close();
        }
        catch (IOException e) {
            throw new BoxfuseException("Unable to extract zip file data into " + targetDir.getAbsolutePath(), e);
        }
    }

    public static String untarGzSingleEntryToString(byte[] tarGz, String entryName) {
        return StringUtils.toUtf8String(ZipUtils.untarGzSingleEntryToByteArray(tarGz, entryName));
    }

    public static byte[] untarGzSingleEntryToByteArray(byte[] tarGz, String entryName) {
        return ZipUtils.untarGzEntriesToByteArrays(tarGz, entryName).get(entryName);
    }

    public static Map<String, byte[]> untarGzEntriesToByteArrays(byte[] tarGz, String ... entryNames) {
        return ZipUtils.untarGzEntriesToByteArrays(tarGz, Arrays.asList(entryNames));
    }

    public static Map<String, byte[]> untarGzEntriesToByteArrays(byte[] tarGz, List<String> entryNames) {
        HashMap<String, byte[]> result = new HashMap<String, byte[]>();
        for (String entryName : entryNames) {
            result.put(entryName, null);
        }
        TarArchiveInputStream tar = null;
        try {
            TarArchiveEntry entry;
            tar = new TarArchiveInputStream(new GZIPInputStream(new ByteArrayInputStream(tarGz)));
            int total = entryNames.size();
            int found = 0;
            while (found < total && (entry = (TarArchiveEntry)tar.getNextEntry()) != null) {
                if (!entryNames.contains(entry.getName())) continue;
                if (!entry.isFile()) {
                    throw new BoxfuseException("Unable to extract tar.gz entry (not a file): " + entry.getName());
                }
                result.put(entry.getName(), IOUtils.copyToByteArrayNoClose(tar, (int)entry.getSize()));
                ++found;
            }
        }
        catch (IOException e) {
            try {
                throw new BoxfuseException("Unable to extract tar.gz entries", e);
            }
            catch (Throwable throwable) {
                IOUtils.close(tar);
                throw throwable;
            }
        }
        IOUtils.close(tar);
        return result;
    }

    public static byte[] gzip(byte[] source) {
        try {
            ByteArrayOutputStream gzip = new ByteArrayOutputStream(source.length);
            IOUtils.copy(source, (OutputStream)new GZIPOutputStream(gzip));
            return gzip.toByteArray();
        }
        catch (Exception e) {
            throw new BoxfuseBugException("Unable to Gzip " + source.length + " bytes", e);
        }
    }

    public static byte[] gzipUtf8(String source) {
        return ZipUtils.gzip(source.getBytes(StandardCharsets.UTF_8));
    }

    public static byte[] gunzip(byte[] source) {
        try {
            ByteArrayOutputStream gunzip = new ByteArrayOutputStream(source.length);
            IOUtils.copy((InputStream)new GZIPInputStream(new ByteArrayInputStream(source)), (OutputStream)gunzip);
            return gunzip.toByteArray();
        }
        catch (Exception e) {
            throw new BoxfuseBugException("Unable to Gzip " + source.length + " bytes", e);
        }
    }

    public static String gunzipUtf8(byte[] source) {
        return new String(ZipUtils.gunzip(source), StandardCharsets.UTF_8);
    }

    public static byte[] tar(String entryName, byte[] entryBytes) {
        try {
            ByteArrayOutputStream output = new ByteArrayOutputStream(entryBytes.length + 1024);
            ArchiveOutputStream out = new ArchiveStreamFactory().createArchiveOutputStream("tar", output);
            TarArchiveEntry tarArchiveEntry = new TarArchiveEntry(entryName);
            tarArchiveEntry.setSize(entryBytes.length);
            tarArchiveEntry.setUserId(0);
            tarArchiveEntry.setGroupId(0);
            tarArchiveEntry.setUserName("root");
            tarArchiveEntry.setGroupName("root");
            tarArchiveEntry.setMode(511);
            out.putArchiveEntry(tarArchiveEntry);
            IOUtils.copyNoClose(entryBytes, (OutputStream)out);
            out.closeArchiveEntry();
            out.close();
            return output.toByteArray();
        }
        catch (Exception e) {
            throw new BoxfuseBugException("Unable to Tar " + entryName + " (" + entryBytes.length + " bytes)", e);
        }
    }

    public static void unxz(File src, File dest) {
        XZCompressorInputStream xzIn = null;
        FileOutputStream out = null;
        try {
            int n;
            xzIn = new XZCompressorInputStream(new BufferedInputStream(new FileInputStream(src)), true);
            out = new FileOutputStream(dest);
            byte[] buffer = new byte[0x100000];
            while (-1 != (n = xzIn.read(buffer))) {
                out.write(buffer, 0, n);
            }
        }
        catch (Exception e) {
            try {
                throw new BoxfuseBugException("Unable to decompress " + src.getAbsolutePath() + " : " + e.getMessage());
            }
            catch (Throwable throwable) {
                IOUtils.close(out);
                IOUtils.close(xzIn);
                throw throwable;
            }
        }
        IOUtils.close(out);
        IOUtils.close(xzIn);
    }

    public static interface EntryFilter {
        public boolean matches(String var1);
    }
}

