/*
 * Decompiled with CFR 0.152.
 */
package com.boxfuse.generator.repository;

import com.boxfuse.base.coordinates.AppCoordinates;
import com.boxfuse.base.coordinates.ImageCoordinates;
import com.boxfuse.base.coordinates.Owner;
import com.boxfuse.base.crypto.CryptoUtils;
import com.boxfuse.base.exception.BoxfuseBugException;
import com.boxfuse.base.exception.BoxfuseException;
import com.boxfuse.base.types.Proxy;
import com.boxfuse.base.types.Version;
import com.boxfuse.base.util.ByteArrayUtils;
import com.boxfuse.base.util.DateUtils;
import com.boxfuse.base.util.HttpUtils;
import com.boxfuse.base.util.IOUtils;
import com.boxfuse.base.util.ProgressBar;
import com.boxfuse.base.util.ShellUtils;
import com.boxfuse.base.util.StopWatch;
import com.boxfuse.base.util.TimeFormat;
import com.boxfuse.base.util.ZipUtils;
import com.boxfuse.generator.app.App;
import com.boxfuse.generator.image.Image;
import com.boxfuse.generator.image.LocalImage;
import com.boxfuse.generator.repository.Vault;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.GeneralSecurityException;
import java.security.Signature;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Repository {
    private static final Logger LOGGER = LoggerFactory.getLogger(Repository.class);
    private final Vault vault;
    private final Owner owner;
    private final Proxy proxy;
    private final boolean insecure;
    private final File dir;
    private SortedMap<ImageCoordinates, Image> localImages;

    public Repository(Vault vault, Owner owner, Proxy proxy, boolean insecure, File dir) {
        this.vault = vault;
        this.owner = owner;
        this.proxy = proxy;
        this.insecure = insecure;
        this.dir = dir;
    }

    public Owner getOwner() {
        return this.owner;
    }

    public Image create(String id, InputStream data) {
        ShellUtils.mkdir(this.dir);
        File file = new File(this.dir.getPath() + "/" + id + ".boxfuse");
        try {
            IOUtils.copy(data, (OutputStream)new FileOutputStream(file));
        }
        catch (IOException e) {
            throw new BoxfuseException("Unable to create Image: " + e.getMessage());
        }
        this.refresh();
        return new LocalImage(file);
    }

    public Image get(String id) {
        return new LocalImage(new File(this.dir.getPath() + "/" + id + ".boxfuse"));
    }

    public OutputStream prepare(String id) {
        ShellUtils.mkdir(this.dir);
        try {
            return new BufferedOutputStream(new FileOutputStream(new File(this.dir.getPath() + "/" + id + ".boxfuse")), 262144);
        }
        catch (FileNotFoundException e) {
            throw new BoxfuseException("Unable to create Image: " + e.getMessage());
        }
    }

    public SortedSet<Image> findAllLocally() {
        TreeSet<Image> images = new TreeSet<Image>();
        images.addAll(this.getLocalImages().values());
        return images;
    }

    public SortedSet<Image> findAllLocally(AppCoordinates coordinates) {
        return this.findAllLocally(coordinates, null);
    }

    public SortedSet<Image> findAllLocally(AppCoordinates coordinates, Version version) {
        TreeSet<Image> images = new TreeSet<Image>(this.findAllLocally());
        Iterator iterator2 = images.iterator();
        while (iterator2.hasNext()) {
            Image image = (Image)iterator2.next();
            if (image.getCoordinates().getAppCoordinates().equals(coordinates)) {
                if (version == null || image.getCoordinates().getVersion().equals(version)) continue;
                iterator2.remove();
                continue;
            }
            iterator2.remove();
        }
        return images;
    }

    public Image findLocally(ImageCoordinates coordinates) {
        if (coordinates == null) {
            return null;
        }
        return (Image)this.getLocalImages().get(coordinates);
    }

    public Image findLocally(int fingerprint) {
        for (Image image : this.getLocalImages().values()) {
            if (image.getMetadata().fingerprint() != fingerprint) continue;
            return image;
        }
        return null;
    }

    public void refresh() {
        this.reloadLocalImages();
    }

    private SortedMap<ImageCoordinates, Image> getLocalImages() {
        if (this.localImages == null) {
            this.reloadLocalImages();
        }
        return this.localImages;
    }

    private void reloadLocalImages() {
        this.localImages = new TreeMap<ImageCoordinates, Image>();
        if (!this.dir.isDirectory()) {
            return;
        }
        File[] files = this.dir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return pathname.isFile() && pathname.canRead() && pathname.getName().endsWith(".boxfuse") && ZipUtils.isZipFile(pathname);
            }
        });
        if (files == null) {
            return;
        }
        for (File file : files) {
            try {
                LocalImage localImage = new LocalImage(file);
                this.localImages.put(localImage.getCoordinates(), localImage);
            }
            catch (Exception e) {
                LOGGER.warn("Unable to read Image: " + file.getAbsolutePath() + " (" + e.getMessage() + ")");
            }
        }
    }

    public ImageCoordinates findInVault(int fingerprint) {
        for (Map.Entry<AppCoordinates, App> app : this.vault.getApps().entrySet()) {
            SortedMap<Version, Integer> images = app.getValue().getImages();
            for (Map.Entry image : images.entrySet()) {
                if (!((Integer)image.getValue()).equals(fingerprint)) continue;
                return new ImageCoordinates(app.getKey(), (Version)image.getKey());
            }
        }
        return null;
    }

    public boolean existsInVault(ImageCoordinates coordinates) {
        App app = this.vault.getApps().get(coordinates.getAppCoordinates());
        return app != null && app.getImages().containsKey(coordinates.getVersion());
    }

    public ImageCoordinates completeImageCoordinates(String coordinates, boolean localOnly, boolean vaultOnly, boolean fail, boolean defaultToLatest) {
        if (!ImageCoordinates.validate(coordinates, this.owner) && !fail) {
            return null;
        }
        ImageCoordinates imageCoordinates = new ImageCoordinates(coordinates, this.owner);
        if (imageCoordinates.getVersion() != null) {
            if (fail && !this.exists(imageCoordinates)) {
                throw new BoxfuseException(imageCoordinates + " doesn't exist => use fuse to create");
            }
            if (fail && localOnly && !this.existsLocally(imageCoordinates)) {
                throw new BoxfuseException(imageCoordinates + " not found locally => pull from Boxfuse Vault to get");
            }
            if (fail && vaultOnly && !this.existsInVault(imageCoordinates)) {
                throw new BoxfuseException(imageCoordinates + " not found in the Boxfuse Vault => push to make it available");
            }
            return imageCoordinates;
        }
        if (defaultToLatest) {
            Version version = this.latestVersion(imageCoordinates.getAppCoordinates(), localOnly, vaultOnly, fail);
            return new ImageCoordinates(imageCoordinates.getAppCoordinates(), version);
        }
        if (fail) {
            throw new BoxfuseException("Incomplete coordinates: " + coordinates + " => also specify the version, not just the app. Example: myapp:1.2.3");
        }
        return null;
    }

    private Version latestVersion(AppCoordinates appCoordinates, boolean localOnly, boolean vaultOnly, boolean fail) {
        SortedMap<Version, Integer> versions;
        Version latestLocalVersion = null;
        if (!vaultOnly) {
            for (ImageCoordinates coordinates : this.getLocalImages().keySet()) {
                if (!coordinates.getAppCoordinates().equals(appCoordinates)) continue;
                latestLocalVersion = this.max(latestLocalVersion, coordinates.getVersion());
            }
        }
        if (localOnly && latestLocalVersion == null) {
            if (fail) {
                throw new BoxfuseException("No Images found for " + appCoordinates + " => use fuse to create one");
            }
            return null;
        }
        App app = this.vault.getApps().get(appCoordinates);
        if (app == null) {
            if (latestLocalVersion != null) {
                return latestLocalVersion;
            }
            if (fail) {
                throw new BoxfuseException("Unknown App: " + appCoordinates + " => log in to your Boxfuse account, create it and try again");
            }
            return null;
        }
        Version latestRemoteVersion = null;
        if (!localOnly && !(versions = app.getImages()).isEmpty()) {
            latestRemoteVersion = versions.lastKey();
        }
        if (latestLocalVersion == null && latestRemoteVersion == null) {
            if (fail) {
                throw new BoxfuseException("No Images found for " + appCoordinates);
            }
            return null;
        }
        return this.max(latestRemoteVersion, latestLocalVersion);
    }

    private Version max(Version version1, Version version2) {
        if (version1 == null) {
            return version2;
        }
        if (version2 == null) {
            return version1;
        }
        return version1.compareTo(version2) >= 0 ? version1 : version2;
    }

    public boolean existsLocally(ImageCoordinates coordinates) {
        return this.getLocalImages().keySet().contains(coordinates);
    }

    public boolean exists(ImageCoordinates coordinates) {
        return this.existsLocally(coordinates) || this.existsInVault(coordinates);
    }

    public void rm(ImageCoordinates imageCoordinates) {
        Image image = this.findLocally(imageCoordinates);
        if (image == null) {
            return;
        }
        File file = image.getFile();
        image.rm();
        LOGGER.info("Removing Image " + image.getCoordinates() + " ...");
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        ShellUtils.deleteFile(file);
        stopWatch.stop();
        LOGGER.info(image.getCoordinates() + " removed in " + TimeFormat.format(stopWatch.getTotalTimeMillis()));
        this.refresh();
    }

    public void rm(AppCoordinates appCoordinates) {
        SortedSet<Image> images = this.findAllLocally(appCoordinates, null);
        LOGGER.info("Removing all locally available Images of " + appCoordinates + " ...");
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        for (Image image : images) {
            LOGGER.info("Removing Image " + image.getCoordinates() + " ...");
            File file = image.getFile();
            image.rm();
            ShellUtils.deleteFile(file);
        }
        stopWatch.stop();
        LOGGER.info("Successfully removed all locally available Images of " + appCoordinates + " in " + TimeFormat.format(stopWatch.getTotalTimeMillis()));
        this.refresh();
    }

    public void push(ImageCoordinates imageCoordinates) {
        if (imageCoordinates == null) {
            throw new BoxfuseException("You must specify the Image to push");
        }
        Image image = this.findLocally(imageCoordinates);
        if (image == null) {
            throw new BoxfuseException("Unknown Image: " + imageCoordinates + " => fuse to create");
        }
        if (image.isLive()) {
            throw new BoxfuseException(imageCoordinates + " has been fused with live reloading and may not be pushed");
        }
        this.push(image);
    }

    public void push(Image image) {
        if (image == null) {
            throw new BoxfuseException("You must specify the Image to push");
        }
        if (!image.getCoordinates().getOwner().equals(this.owner)) {
            throw new BoxfuseException("You can only push your own Images to the Boxfuse Vault.");
        }
        if (!this.vault.isOnline()) {
            throw new BoxfuseException("You must be online to push => check your internet connection and try again.");
        }
        if (this.existsInVault(image.getCoordinates())) {
            LOGGER.info(image.getCoordinates() + " already exists in the Boxfuse Vault. No need to push.");
            return;
        }
        try {
            byte[] imageBytes;
            LOGGER.info("Pushing " + image.getCoordinates() + " ...");
            Map<String, Object> result = this.vault.push(image.getId(), image.getCoordinates(), ByteArrayUtils.toHex(image.getSignature()), image.getSize(), DateUtils.toUtcDateTimeString(image.getGeneratedUtc()), image.getManifest());
            String hexKey = (String)result.get("key");
            List purgedVersion = (List)result.get("purged");
            for (String v : purgedVersion) {
                LOGGER.warn("Removed " + v + " from Vault as it was full => upgrade to a larger plan to retain more Images");
            }
            byte[] clear = IOUtils.copyToByteArray(image.getFile());
            if (hexKey == null) {
                imageBytes = clear;
            } else {
                byte[] key = ByteArrayUtils.fromHex(hexKey);
                imageBytes = CryptoUtils.aesEncrypt(clear, key);
            }
            this.uploadAsMultipart(image, imageBytes);
        }
        catch (BoxfuseException e) {
            throw e;
        }
        catch (Exception e) {
            throw new BoxfuseException("Unable to push " + image.getCoordinates(), e);
        }
    }

    private void uploadAsMultipart(final Image image, byte[] encrypted) throws Exception {
        final List<byte[]> parts = ByteArrayUtils.split(encrypted, 0x500000);
        ArrayList<Map<String, Object>> partsInfo = new ArrayList<Map<String, Object>>();
        for (byte[] part : parts) {
            HashMap<String, Object> partInfo = new HashMap<String, Object>();
            partInfo.put("size", part.length);
            partInfo.put("md5", ByteArrayUtils.toHex(CryptoUtils.md5Digest(part)));
            partsInfo.add(partInfo);
        }
        Map<String, Object> partsResult = this.vault.pushParts(image.getId(), image.getCoordinates(), encrypted.length, partsInfo);
        final List partUrls = (List)partsResult.get("partUrls");
        final ProgressBar progressBar = new ProgressBar(encrypted.length);
        final String[] partETags = new String[parts.size()];
        ExecutorService executorService = Executors.newFixedThreadPool(4);
        try {
            for (int i2 = 0; i2 < partUrls.size(); ++i2) {
                final int k = i2;
                final int num = i2 + 1;
                final String md5 = (String)((Map)partsInfo.get(k)).get("md5");
                executorService.submit(new Runnable(){

                    @Override
                    public void run() {
                        LOGGER.debug("Pushing " + image.getCoordinates() + " (part " + num + "/" + partUrls.size() + ") ...");
                        partETags[k] = HttpUtils.putWithProgress(progressBar, num, (String)partUrls.get(k), (byte[])parts.get(k), md5, Repository.this.proxy, Repository.this.insecure);
                        LOGGER.debug("Pushed part " + num + " -> " + partETags[k]);
                    }
                });
            }
            executorService.shutdown();
            executorService.awaitTermination(60L, TimeUnit.MINUTES);
        }
        catch (Exception e) {
            executorService.shutdownNow();
            throw e;
        }
        LOGGER.info("Verifying " + image.getCoordinates() + " ...");
        this.vault.verify(image.getId(), Arrays.asList(partETags));
    }

    public Image pull(ImageCoordinates imageCoordinates) {
        if (imageCoordinates == null) {
            throw new BoxfuseException("You must specify the Image to pull");
        }
        if (this.existsLocally(imageCoordinates)) {
            LOGGER.info(imageCoordinates + " already exists locally. No need to pull.");
            return this.findLocally(imageCoordinates);
        }
        if (!imageCoordinates.getOwner().equals(this.owner)) {
            throw new BoxfuseException("You can only pull your own Images from the Boxfuse Vault.");
        }
        if (!this.vault.isOnline()) {
            throw new BoxfuseException("You must be online to pull => check your internet connection and try again.");
        }
        if (!this.existsInVault(imageCoordinates)) {
            throw new BoxfuseException("Unknown Image: " + imageCoordinates);
        }
        try {
            byte[] clear;
            LOGGER.info("Pulling " + imageCoordinates + " ...");
            Map<String, Object> result = this.vault.pull(imageCoordinates);
            String id = (String)result.get("id");
            String url = (String)result.get("url");
            byte[] imageBytes = HttpUtils.downloadWithProgress(url, this.proxy, this.insecure);
            byte[] publicKey = ByteArrayUtils.fromHex((String)result.get("publicKey"));
            LOGGER.info("Verifying " + imageCoordinates + " ...");
            String hexKey = (String)result.get("key");
            if (hexKey == null) {
                clear = imageBytes;
            } else {
                byte[] key = ByteArrayUtils.fromHex(hexKey);
                clear = CryptoUtils.aesDecrypt(imageBytes, key);
            }
            byte[] signatureBytes = ZipUtils.unzipSingleEntryToByteArray(clear, id + ".sig");
            byte[] manifestBytes = ZipUtils.unzipSingleEntryToByteArray(clear, id + ".manifest");
            byte[] diskBytes = ZipUtils.unzipSingleEntryToByteArray(clear, id + ".disk.gz");
            diskBytes = diskBytes == null ? ZipUtils.unzipSingleEntryToByteArray(clear, id + ".disk") : ZipUtils.gunzip(diskBytes);
            try {
                Signature signature = Signature.getInstance("SHA512withRSA");
                signature.initVerify(CryptoUtils.rsaPublicKey(publicKey));
                signature.update(manifestBytes);
                signature.update(diskBytes);
                if (!signature.verify(signatureBytes)) {
                    throw new BoxfuseException("Invalid Image signature for: " + imageCoordinates);
                }
            }
            catch (GeneralSecurityException e) {
                throw new BoxfuseBugException("Unable to verify Image signature", e);
            }
            return this.create(id, new ByteArrayInputStream(clear));
        }
        catch (Exception e) {
            throw new BoxfuseException("Unable to pull " + imageCoordinates, e);
        }
    }

    public Map<String, Object> rmFromVault(ImageCoordinates coordinates) {
        if (!coordinates.getOwner().equals(this.owner)) {
            throw new BoxfuseException("You can only remove your own Images from the Boxfuse Vault.");
        }
        if (!this.vault.isOnline()) {
            throw new BoxfuseException("You must be online to run rm against the Boxfuse Vault => check your internet connection and try again.");
        }
        if (!this.existsInVault(coordinates)) {
            LOGGER.info(coordinates + " doesn't exist in the Boxfuse Vault. No need to remove it.");
            return null;
        }
        try {
            LOGGER.info("Removing " + coordinates + " from the Boxfuse Vault ...");
            return this.vault.rm(coordinates.getAppCoordinates(), coordinates.getVersion());
        }
        catch (BoxfuseException e) {
            throw e;
        }
        catch (Exception e) {
            throw new BoxfuseException("Unable to purge " + coordinates, e);
        }
    }

    public Map<String, Object> rmFromVault(AppCoordinates appCoordinates) {
        if (!appCoordinates.getOwner().equals(this.owner)) {
            throw new BoxfuseException("You can only remove your own Images from the Boxfuse Vault.");
        }
        if (!this.vault.isOnline()) {
            throw new BoxfuseException("You must be online to run rm against the Boxfuse Vault => check your internet connection and try again.");
        }
        if (this.vault.getApps().get(appCoordinates) == null) {
            LOGGER.info(appCoordinates + " doesn't exist in the Boxfuse Vault. No need to remove its Images.");
            return null;
        }
        try {
            LOGGER.info("Removing all Images of " + appCoordinates + " from the Boxfuse Vault ...");
            return this.vault.rm(appCoordinates, null);
        }
        catch (BoxfuseException e) {
            throw e;
        }
        catch (Exception e) {
            throw new BoxfuseException("Unable to remove all Images of " + appCoordinates, e);
        }
    }
}

