/*
 * Decompiled with CFR 0.152.
 */
package com.boxfuse.base.util;

import com.boxfuse.base.exception.BoxfuseException;
import com.boxfuse.base.util.ExceptionUtils;
import com.boxfuse.base.util.IOUtils;
import com.boxfuse.base.util.OSUtils;
import com.boxfuse.base.util.StringUtils;
import com.boxfuse.base.util.ThreadUtils;
import com.boxfuse.base.util.Tuple3;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ShellUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(ShellUtils.class);

    private ShellUtils() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static <T> T timedCall(Callable<T> c, long timeout, TimeUnit timeUnit) throws InterruptedException, ExecutionException, TimeoutException {
        FutureTask<T> task = new FutureTask<T>(c);
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        try {
            executorService.execute(task);
            T t = task.get(timeout, timeUnit);
            return t;
        }
        finally {
            executorService.shutdown();
        }
    }

    public static void mkdir(String dir) {
        ShellUtils.mkdir(new File(dir));
    }

    public static void mkdir(File dir) {
        dir.mkdirs();
    }

    public static File mkTempDir() {
        try {
            File temp = Files.createTempDirectory("boxtmp", new FileAttribute[0]).toFile();
            Runtime.getRuntime().addShutdownHook(new Thread(() -> ShellUtils.deleteDir(temp)));
            temp.deleteOnExit();
            return temp;
        }
        catch (IOException e) {
            throw new BoxfuseException("Could not create temp directory", e);
        }
    }

    public static String findCommandInDirs(String command, String ... dirs) {
        for (String dir : dirs) {
            File file;
            if (dir == null || !(file = new File(dir, command)).isFile() || !file.canExecute()) continue;
            LOGGER.debug("Found " + command + " in " + dir);
            return file.getAbsolutePath();
        }
        return null;
    }

    public static String execPowerShellCommand(String cmd) {
        return ShellUtils.execCommandWithRetries(0, new ArrayList<String>(), new HashMap<String, Integer>(), "powershell", cmd);
    }

    public static String execPowerShellCommand(List<String> retryOutputs, String cmd) {
        return ShellUtils.execCommandWithRetries(3, retryOutputs, new HashMap<String, Integer>(), "powershell", cmd);
    }

    public static void execInteractiveCommand(String ... commands) {
        ShellUtils.execInteractiveCommand(null, commands);
    }

    public static void execInteractiveCommand(Map<String, String> envvars, String ... commands) {
        String commandStr = StringUtils.arrayToDelimitedString(commands, " ");
        ProcessBuilder processBuilder = new ProcessBuilder(commands).inheritIO();
        if (envvars != null) {
            processBuilder.environment().putAll(envvars);
        }
        try {
            Process process = processBuilder.start();
            int exitCode = process.waitFor();
            if (exitCode != 0) {
                throw new BoxfuseException("Executing " + commandStr + " failed! (Return code: " + exitCode + ")");
            }
        }
        catch (IOException | InterruptedException e) {
            throw new BoxfuseException("Executing " + commandStr + " failed: " + e.getMessage());
        }
    }

    public static String execCommand(String command, String ... args) {
        return ShellUtils.execCommandWithRetries(0, new ArrayList<String>(), new HashMap<String, Integer>(), command, args);
    }

    public static String execCommandWithRetries(int maxRetries, List<String> retryOutputs, Map<String, Integer> notFailedOutputs, String command, String ... args) {
        ArrayList<String> commands = new ArrayList<String>();
        commands.add(command);
        commands.addAll(Arrays.asList(args));
        String commandStr = command + " " + StringUtils.arrayToDelimitedString(args, " ");
        Tuple3 result = null;
        for (int i = 0; i <= maxRetries; ++i) {
            ProcessBuilder processBuilder = new ProcessBuilder(commands);
            try {
                Process process = processBuilder.start();
                try {
                    result = ShellUtils.timedCall(() -> {
                        String stdOut = IOUtils.copyToString(process.getInputStream());
                        String stdErr = IOUtils.copyToString(process.getErrorStream());
                        return Tuple3.of(stdOut, stdErr, process.waitFor());
                    }, 30L, TimeUnit.SECONDS);
                }
                catch (TimeoutException te) {
                    process.destroy();
                    if (i < maxRetries) {
                        LOGGER.warn("Timeout executing command: " + commandStr + " => retrying ...");
                        continue;
                    }
                    throw te;
                }
            }
            catch (Exception e) {
                if (i < maxRetries) {
                    LOGGER.debug("Unable to execute command: " + commandStr + " => retrying ...", (Throwable)e);
                    ThreadUtils.sleep(1000L * (1L << i));
                    continue;
                }
                throw new BoxfuseException("Unable to execute command: " + commandStr, e);
            }
            if (ShellUtils.succeeded(notFailedOutputs, (String)result.getE1(), (String)result.getE2(), (Integer)result.getE3())) {
                result = Tuple3.of(result.getE1(), result.getE2(), 0);
                break;
            }
            if (i >= maxRetries || !ShellUtils.shouldRetry(retryOutputs, (String)result.getE1(), (String)result.getE2(), result.getE3())) break;
            LOGGER.debug("Command failed (exit code -> " + result.getE3() + "): " + commandStr + " => Retrying...");
            ThreadUtils.sleep(1000L * (1L << i));
        }
        if (result != null) {
            if ((Integer)result.getE3() != 0) {
                throw new BoxfuseException("Command failed (exit code -> " + result.getE3() + "): \n" + commandStr + "\nStandard Output:\n" + (String)result.getE1() + "\nStandard Error:\n" + (String)result.getE2());
            }
            return (String)result.getE1();
        }
        return "";
    }

    private static boolean succeeded(Map<String, Integer> notFailedOutputs, String stdOut, String stdErr, int result) {
        if (result == 0) {
            return true;
        }
        for (Map.Entry<String, Integer> entry : notFailedOutputs.entrySet()) {
            if (result != entry.getValue() || !stdOut.contains(entry.getKey()) && !stdErr.contains(entry.getKey())) continue;
            return true;
        }
        return false;
    }

    private static boolean shouldRetry(List<String> retryOutputs, String stdOut, String stdErr, int result) {
        if (result == 0) {
            return false;
        }
        for (String retryOutput : retryOutputs) {
            if (!stdOut.contains(retryOutput) && !stdErr.contains(retryOutput)) continue;
            return true;
        }
        return false;
    }

    public static void execCommandAsync(String ... commands) {
        try {
            if (OSUtils.runningOnWindows()) {
                ArrayList<String> cmdList = new ArrayList<String>();
                cmdList.addAll(Arrays.asList("cmd", "/c", "start", "\"\""));
                cmdList.addAll(Arrays.asList(commands));
                Runtime.getRuntime().exec(cmdList.toArray(new String[0]));
            } else {
                Runtime.getRuntime().exec(commands);
            }
        }
        catch (IOException e) {
            throw new BoxfuseException("Unable to execute command: " + StringUtils.arrayToDelimitedString(commands, " ") + " (" + e.getMessage() + ")");
        }
    }

    public static void deleteDir(File dir) {
        if (dir == null) {
            return;
        }
        File[] files = dir.listFiles();
        if (files != null) {
            for (File f : files) {
                if (f.isDirectory()) {
                    ShellUtils.deleteDir(f);
                    continue;
                }
                f.delete();
            }
        }
        dir.delete();
    }

    public static void createDirHardLink(File hardLink, File destinationDir) {
        if (OSUtils.runningOnWindows()) {
            ShellUtils.execCommand("cmd", "/c", "mklink", "/j", hardLink.getAbsolutePath(), destinationDir.getAbsolutePath());
        } else if (OSUtils.runningOnMac()) {
            ShellUtils.execCommand("ln", destinationDir.getAbsolutePath(), hardLink.getAbsolutePath());
        } else {
            ShellUtils.execCommand("ln", destinationDir.getAbsolutePath(), hardLink.getAbsolutePath());
        }
    }

    public static boolean isRunning(String process) {
        if (OSUtils.runningOnWindows()) {
            return ShellUtils.execCommand("tasklist", "/FI", "IMAGENAME eq " + process).contains(process);
        }
        return ShellUtils.execCommand("ps", "ax").contains(process);
    }

    public static boolean isRunning(int pid) {
        String pidStr = "" + pid;
        if (OSUtils.runningOnWindows()) {
            return ShellUtils.execCommand("tasklist", "/FI", "PID eq " + pid).contains(pidStr);
        }
        HashMap<String, Integer> notFailedOutputs = new HashMap<String, Integer>();
        notFailedOutputs.put("PID", 1);
        return ShellUtils.execCommandWithRetries(0, new ArrayList<String>(), notFailedOutputs, "ps", "-p", pidStr).contains(pidStr);
    }

    public static void kill(String process) {
        if (OSUtils.runningOnWindows()) {
            ShellUtils.execCommand("taskkill", "/IM", process, "/F");
        } else {
            ShellUtils.execCommand("pkill", process);
        }
    }

    public static void kill(int pid) {
        String pidStr = "" + pid;
        if (OSUtils.runningOnWindows()) {
            ShellUtils.execCommand("taskkill", "/PID", pidStr, "/F");
        } else {
            ShellUtils.execCommand("kill", "-9", pidStr);
        }
    }

    public static void deleteFile(File file) {
        if (file == null) {
            return;
        }
        int retries = 3;
        while (!file.delete()) {
            if (retries-- == 0) {
                file.deleteOnExit();
                LOGGER.debug("Unable to delete " + file.getAbsolutePath());
                return;
            }
            ThreadUtils.sleep(1000L);
        }
    }

    public static void deleteDirHardLink(File hardLink) {
        if (hardLink == null) {
            return;
        }
        if (OSUtils.runningOnWindows()) {
            ShellUtils.execCommand("cmd", "/c", "rmdir", hardLink.getAbsolutePath());
        } else {
            ShellUtils.execCommand("rm", hardLink.getAbsolutePath());
        }
    }

    public static boolean isDirEmpty(File dir) {
        String[] files = dir.list();
        return files == null || files.length == 0;
    }

    public static boolean isRoot() {
        return "root".equals(ShellUtils.getUserName());
    }

    public static String getUserName() {
        return System.getProperty("user.name");
    }

    public static int getUserId() {
        return Integer.parseInt(ShellUtils.execCommand("id", "-u").trim());
    }

    public static int getGroupId() {
        return Integer.parseInt(ShellUtils.execCommand("id", "-g").trim());
    }

    public static void renameFile(File src, File dest) {
        int retries = 3;
        while (!src.renameTo(dest)) {
            if (retries-- == 0) {
                throw new BoxfuseException("Unable to rename " + src.getAbsolutePath() + " to " + dest.getAbsolutePath());
            }
            ThreadUtils.sleep(1000L);
            if (!dest.exists()) continue;
            ShellUtils.deleteFile(dest);
        }
    }

    public static String getFileExtension(String name) {
        try {
            return name.substring(name.lastIndexOf(".") + 1);
        }
        catch (Exception e) {
            return "";
        }
    }

    public static String getFileNameWithoutExtension(String name) {
        try {
            return name.substring(0, name.lastIndexOf("."));
        }
        catch (Exception e) {
            return "";
        }
    }

    public static boolean executableExists(String executable) {
        String command = OSUtils.runningOnWindows() ? "where" : "which";
        try {
            ShellUtils.execCommand(command, executable);
            return true;
        }
        catch (Exception e) {
            LOGGER.debug("Unable to find " + executable + ": " + ExceptionUtils.getRootCause(e).getMessage());
            return false;
        }
    }
}

